<% '***************************************************************************
   '* ASP Football Pool                                                       *
   '*                                                                         *
   '* Copyright 2000 by Mike Hall                                             *
   '* Please see http://www.brainjar.com for documentation and terms of use.  *
   '***************************************************************************

   '***************************************************************************
   '* Constant definitions.                                                   *
   '***************************************************************************

   'Administrator username.

   Const ADMIN_USERNAME = "Admin"

   'Minimum password length

   Const PASSWORD_MIN_LEN = 6

   'Time zone difference between host server and database schedule (Eastern).

   Const TIME_ZONE_DIFF = -3

   'Cost of weekly bet.

   Const BET_AMOUNT = 5

   Const TIE_STR = "Tie"

   '***************************************************************************
   '* Global variables.                                                       *
   '***************************************************************************

   dim DbConn

   '***************************************************************************
   '* Functions and subroutines.                                              *
   '***************************************************************************

   '---------------------------------------------------------------------------
   ' OpenDB(): Opens the database connection (global variable 'DbConn').
   '---------------------------------------------------------------------------

   sub OpenDB()

     set DbConn = Server.CreateObject("ADODB.Connection")
     dbDir = Server.MapPath("./data/NFL.mdb")
     DbConn.Open "DBQ="& dbDir _
       &";Driver={Microsoft Access Driver (*.mdb)};DriverId=25;FIL=MS Access;"

   end sub

   '---------------------------------------------------------------------------
   ' CurrentDateTime(): Returns a date object representing the current date
   ' and time adjusted to the time zone used in the database.
   '---------------------------------------------------------------------------

   function CurrentDateTime()

     CurrentDateTime = DateAdd("h", -TIME_ZONE_DIFF, _
       CDate(Date() & " " & Time()))

   end function

   '---------------------------------------------------------------------------
   ' CurrentWeek(): Returns the current week number based on the current
   ' (time zone adjusted) date.
   '---------------------------------------------------------------------------

   function CurrentWeek()

     dim sql, datenow, rs, found

     CurrentWeek = 0
     datenow = CurrentDateTime
     sql = "select Week, Date from Schedule order by Date"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       found = false
       do while not rs.EOF and not found
         if DateDiff("d", DateValue(rs.Fields("Date").Value), datenow) <= 0 then
           found = true
           currentWeek = rs.Fields("Week").Value
         end if
         rs.MoveNext
       loop
     end if
     if CurrentWeek = 0 then
       CurrentWeek = NumberOfWeeks()
     end if

   end function

   '---------------------------------------------------------------------------
   ' NumberOfUsers(): Returns the number of users in the database excluding the
   ' administrator.
   '---------------------------------------------------------------------------

   function NumberOfUsers()

     dim sql, rs

     NumberOfUsers = 0
     sql = "select count(*) as Total from Users" _
        & " where Username <> '" & ADMIN_USERNAME & "'"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       NumberOfUsers = rs.Fields("Total").Value
     end if

   end function

   '---------------------------------------------------------------------------
   ' NumberOfGames(week): Returns the number of games scheduled for week given.
   '---------------------------------------------------------------------------

   function NumberOfGames(week)

     dim sql, rs

     NumberOfGames = 0
     sql = "select count(*) as Total from Schedule where Week = " & week
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       NumberOfGames = rs.Fields("Total").Value
     end if

   end function

   '---------------------------------------------------------------------------
   ' NumberOfWeeks(): Returns the total number of weeks in the schedule.
   '---------------------------------------------------------------------------

   function NumberOfWeeks()

     dim sql, rs

     NumberOfWeeks = 0
     sql = "select max(Week) as Total from Schedule"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       NumberOfWeeks = rs.Fields("Total").Value
     end if

   end function

   '---------------------------------------------------------------------------
   ' UserScore(username, week): Returns the number of correct picks for the
   ' given user in the given week. If the user did not enter any picks that
   ' week, a null string is returned.
   '---------------------------------------------------------------------------

   function UserScore(username, week)

     dim sql, rs, n

     UserScore = ""

     'Return a null string if no picks were made.

     sql = "select count(*) as total" _
        & " from Picks, Schedule" _
        & " where Username = '" & username & "'" _
        & " and Schedule.Week = " & week _
        & " and Picks.GameID = Schedule.GameID"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       if rs.Fields("total").Value = 0 then
         exit function
       end if
     else
       exit function
     end if

     'Otherwise, return the number of correct picks.
   
     sql = "select count(*) as score" _
        & " from Picks, Schedule" _
        & " where Username = '" & username & "'" _
        & " and Schedule.Week = " & week _
        & " and Picks.GameID = Schedule.GameID" _
        & " and Picks.Pick = Schedule.Result" _
        & " and Schedule.Result <> ''"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       UserScore = rs.Fields("score").Value
     end if

   end function

   '---------------------------------------------------------------------------
   ' UserTBGuess(username, week): Returns the given user's tie breaker
   ' guess for the given week.
   '---------------------------------------------------------------------------

   function UserTBGuess(username, week)

     dim sql, rs, n

     UserTBGuess = ""

     sql = "select TBPoints" _
        & " from TieBreaker" _
        & " where Username = '" & username & "'" _
        & " and Week = " & week
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       UserTBGuess = rs.Fields("TBPoints").Value
     end if

   end function

   '---------------------------------------------------------------------------
   ' TBPointTotal(week): Returns the actual point total of the Monday Night
   ' game for the given week. If those scores are not available, a null string
   ' is returned.
   '---------------------------------------------------------------------------

   function TBPointTotal(week)

     dim sql, rs, n

     TBPointTotal = ""
     sql = "select VisitorScore + HomeScore as total from Schedule" _
        & " where Week = " & week _
        & " order by Date desc"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       n = rs.Fields("total").Value
       if not IsNumeric(n) then
         exit function
       end if
       TBPointTotal = n
     end if

   end function

   '---------------------------------------------------------------------------
   ' NumberOfEntries(week): Returns the total number of user's who made picks
   ' for the given week.
   '---------------------------------------------------------------------------

   function NumberOfEntries(week)

     dim sql, rs

     NumberOfEntries = 0
     sql = "select distinct Username" _
         & " from Picks, Schedule" _
         & " where Week = " & week _
         & " and Picks.GameID = Schedule.GameID"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       do while not rs.EOF
         NumberOfEntries = NumberOfEntries + 1
         rs.MoveNext
       loop
     end if

   end function

   '---------------------------------------------------------------------------
   ' WinnersList(week): Compares scores and tie breakers to determine the
   ' winner or winners for the given week. An array of the winners user names
   ' is returned. If the week has not been concluded, or no players entered
   ' that week, a null string is returned instead.
   '---------------------------------------------------------------------------

   function WinnersList(week)

     dim sql, rs
     dim tbActual
     dim highScore, lowDiff
     dim username, score, tb, diff
     dim list

     WinnersList = ""

     'Get point total for the tie breaker, exit if not available.

     tbActual = TBPointTotal(week)
     if not IsNumeric(tbActual) then
       exit function
     end if

     'Initialize current high score and tie breaker difference.

     highScore = -1
     lowDiff   =  0

     'Check all users.

     sql = "select Username" _
        & " from Users" _
        & " where Username <> '" & ADMIN_USERNAME & "'"
     set rs = DbConn.Execute(sql)
     if not (rs.BOF and rs.EOF) then
       do while not rs.EOF
         username = rs.Fields("Username").Value
         score = UserScore(username, week)
         tb = UserTBGuess(username, week)

         if IsNumeric(score) and IsNumeric(tb) then

           'Compare user's score to the current highest.

           diff = Abs(tbActual - tb)
           if score > highScore or (score = highScore and diff < lowDiff) then
             redim list(0)
             list(0) = username
             highScore = score
             lowDiff = diff
           elseif score = highScore and diff = lowDiff then
             redim preserve list(UBound(list) + 1)
             list(UBound(list)) = username
           end if           
         end if

         rs.MoveNext
       loop
     end if

     WinnersList = list

   end function

   '---------------------------------------------------------------------------
   ' OpenDates(week): Returns a string of names for teams that do not have a
   ' game scheduled for given week.
   '---------------------------------------------------------------------------

   function OpenDates(week)

     dim sql, rs, rs2
     dim id, team

     OpenDates = ""

     sql = "select * from Teams" _
        & " order by City, Name"
     set rs = DbConn.Execute(sql)
     do while not rs.EOF
       id = rs.Fields("TeamID").Value
       sql = "select count(*) as total" _
          & " from Schedule" _
          & " where (HomeID = '" & id & "'" _
          & " or VisitorID = '" & id & "')" _
          & " and week = " & week
       set rs2 = DbConn.Execute(sql)
       if rs2.Fields("total").Value = 0 then
         team = rs.Fields("City").Value
         if rs.Fields("DisplayName").Value <> "" then
           team = rs.Fields("DisplayName").Value
         end if
         if OpenDates <> "" then
           OpenDates = OpenDates & ", "
         end if
         OpenDates = OpenDates & team
       end if
       rs.MoveNext
     loop

   end function

   '---------------------------------------------------------------------------
   ' FormatTime(dt): Returns a formatted string based on the time value of the
   ' given date.
   '---------------------------------------------------------------------------

   function FormatTime(dt)

     dim hh, mm, symbol

     hh = Hour(dt)
     mm = Minute(dt)
     if mm < 10 then
     mm = "0" & mm
     end if
     if hh >= 12 then
       symbol = "pm"
     else
       symbol = "am"
     end if
     if hh >= 13 then
       hh = hh - 12
     end if
     FormatTime = hh & ":" & mm & " " & symbol

   end function

   '---------------------------------------------------------------------------
   ' FormatDate(dt): Returns a formatted string based on the date value of the
   ' given date.
   '---------------------------------------------------------------------------

   function FormatDate(dt)

     FormatDate = Month(dt) & "/" & Day(dt)

   end function

   '---------------------------------------------------------------------------
   ' FormatAmount(n): Returns a formatted string based on the given number as
   ' a dollar amount.
   '---------------------------------------------------------------------------

   function FormatAmount(n)

     FormatAmount = "&nbsp;"
     if IsNumeric(n) then
       FormatAmount = FormatCurrency(n, 2, true, false, true)
     end if

   end function

   '---------------------------------------------------------------------------
   ' FormatPercent(n): Returns a formatted string based on the given number as
   ' a percentage.
   '---------------------------------------------------------------------------

   function FormatPercent(n)

     FormatPercent = "&nbsp;"
     if IsNumeric(n) then
       FormatPercent = FormatNumber(Round(n, 3), 3, true)
     end if

   end function

   '---------------------------------------------------------------------------
   ' CreateSalt(): Returns an eight-character random salt string. Used for
   ' generating the hash for passwords.
   '---------------------------------------------------------------------------

   function CreateSalt()

     dim i, n, chars

     CreateSalt = ""
     Randomize
     chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
     for i = 1 to 8
       n = Int(Len(chars) * Rnd()) + 1
       CreateSalt = CreateSalt & Mid(chars, n, 1)
     next

   end function

   '---------------------------------------------------------------------------
   ' Hash(str): Returns a 40-character hash value for the given string. Used
   ' for storing passwords.
   '---------------------------------------------------------------------------

   function Hash(str)

     'Call the JScript function to calculate the SHA-1 message digest for the
     'string.

     Hash = SecureHash(str)

   end function

   '---------------------------------------------------------------------------
   ' ErrorMessage(msg): Outputs 'msg' in the error style.
   '---------------------------------------------------------------------------

   sub ErrorMessage(msg)

     Response.Write("<br><span class=""error"">" & msg & "</span>")

   end sub

   '---------------------------------------------------------------------------
   ' InfoMessage(msg): Outputs 'msg' in the info style.
   '---------------------------------------------------------------------------

   sub InfoMessage(msg)

     Response.Write("<br><span class=""info"">" & msg & "</span>")

   end sub %>

<SCRIPT LANGUAGE="JScript" RUNAT="Server">

   // Note change in scripting language.

   /***************************************************************************
    * SecureHash(s): Calculates the message digest for the given string using *
    * the SHA-1 algorithm.                                                    *
    **************************************************************************/

   function SecureHash(s) {

     var bitLen, n, w;
     var k, buffer, f;
     var h0, h1, h2, h3, h4;
     var a, b, c, d, e, temp;
     var i, j;

     // Pad the input message per the SHA-1 specification.

     bitLen = s.length * 8;
     n = 512 - (bitLen % 512);
     if (n <= 65)
       n += 512;
     s += String.fromCharCode(0x80);
     n -= 8;
     for (i = 0; i < n / 8; i++)
       s += String.fromCharCode(0x00);

     // Convert padded message to an array of 32-bit integers, note conversion
     // to little endian.

     w = new Array();
     for (i = 0; i < s.length; i += 4) {
       n = 0;
       for (j = 0; j < 4; j++)
         n = (n << 8) + s.charCodeAt(i + j);
       w[w.length] = n;
     }

     // Set last word to the original bit length of the message.

     w[w.length - 1] = bitLen;

     // Initialize hash values, constants and buffer.

     h0 = 0x67452301;
     h1 = 0xEFCDAB89;
     h2 = 0x98BADCFE;
     h3 = 0x10325476;
     h4 = 0xC3D2E1F0;

     k = new Array(80);
     for (i = 0; i < 80; i++)
      if (i < 20)
        k[i] = 0x5A827999;
      else if (i < 40)
        k[i] = 0x6ED9EBA1;
      else if (i < 60)
        k[i] = 0x8F1BBCDC;
      else
        k[i] = 0xCA62C1D6;

     buffer = new Array(80);

     // Process word array in 512-bit (16-word) blocks.

     n = w.length / 16;
     for (i = 0; i < n; i ++) {

       // Initialize 80-word buffer using the current block.

       for (j = 0; j < 80; j++)
         if (j < 16)
           buffer[j] = w[i * 16 + j];
         else
           buffer[j] = rol(buffer[j - 3] ^ buffer[j - 8] ^ buffer[j - 14] ^
                           buffer[j - 16], 1);

       a = h0; b = h1; c = h2; d = h3; e = h4;

       // Hash the block.

       for (j = 0; j < 80; j++) {
         temp = rol(a, 5);
         if (j < 20)
           f = (b & c) | ((~b) & d);
         else if (j < 40)
           f = b ^ c ^ d;
         else if (j < 60)
           f = (b & c) | (b & d) | (c & d);
         else
           f = b ^ c ^ d;
         temp = add(temp, f);
         temp = add(temp, e);
         temp = add(temp, buffer[j]);
         temp = add(temp, k[j]);
         e = d; d = c; c = rol(b, 30); b = a; a = temp;
       }

       // Update hash values.

       h0 = add(h0, a);
       h1 = add(h1, b);
       h2 = add(h2, c);
       h3 = add(h3, d);
       h4 = add(h4, e);
     }

     // Format hash values and return as message digest.

     return hex(h0) + hex(h1) + hex(h2) + hex(h3) + hex(h4);
   }

   // Helper functions for the Secure Hash function.

   function rol(x, n) {

     // Left circular shift for a 32-bit integer.

     return (x << n) | (x >>> (32 - n));
   }

   function add(x, y) {

     // Add two 32-bit integers, wrapping at 2^32.

     return ((x & 0x7FFFFFFF) + (y & 0x7FFFFFFF)) ^
             (x & 0x80000000) ^ (y & 0x80000000)
   }

   function hex(n) {

     var hexDigits = "0123456789ABCDEF";
     var i, s;

     // Format a 32-bit integer as a hexadecimal string.

     s = "";
     for (i = 7; i >= 0; i--)
       s += hexDigits.charAt((n >> (i * 4)) & 0x0F);

     return s;
   }

</SCRIPT>

